Imports EJRandomOrgConsumer.StructuresAndEnums
Imports System.ComponentModel

''' <summary>
''' Provides methods and properties for consuming the HTTP service at random.org that generates random strings.
''' </summary>
''' <remarks></remarks>
Public Class StringGenerator
    Inherits BaseGenerator


    Private Const MaxQuantity As Short = 10000
    Private Const MaxStringLength As Byte = 20

    Private _URL As String = "http://www.random.org/strings/"
    Private _Quantity As Short = 1
    Private _StringLength As Byte = 1
    Private _AllowNumbers As Boolean = True
    Private _AllowUppercaseAlpha As Boolean = True
    Private _AllowLowercaseAlpha As Boolean = True
    Private _EnsureUnique As Boolean = False


#Region " Properties "


    ''' <summary>
    ''' Gets or sets a value specifying the URL providing the service.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue("http://www.random.org/strings/"), Description("Gets or sets a value specifying the URL providing the service.")> _
     Public Property URL() As String
        Get
            Return _URL
        End Get
        Set(ByVal value As String)
            _URL = value
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying the number of items in the array returned by the GetIntegers() function.
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(1), Description("Gets or sets a value specifying the number of items in the array returned by the GetIntegers() function.")> _
    Public Property Quantity() As Short
        Get
            Return _Quantity
        End Get
        Set(ByVal value As Short)
            If _Quantity < 0 Then
                Throw New Exception("The minimum allowed quantity is 1.")
            ElseIf _Quantity > MaxQuantity Then
                Throw New Exception("The minimum allowed quantity is " & MaxQuantity.ToString("#,###") & ".")
            Else
                _Quantity = value
            End If
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying the length of the randomly-generated string(s).
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(1), Description("Gets or sets a value specifying the length of the randomly-generated string(s).")> _
    Public Property StringLength() As Byte
        Get
            Return _StringLength
        End Get
        Set(ByVal value As Byte)
            If value < 1 Then
                Throw New Exception("The minimum allowed string length is 1 character.")
            ElseIf value > MaxStringLength Then
                Throw New Exception("The maximum allowed string length is " & MaxStringLength.ToString("#,###") & " characters.")
            Else
                _StringLength = value
            End If
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying whether numeric characters are allowed to appear in the randomly-generated string(s).
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(True), Description("Gets or sets a value specifying whether numeric characters are allowed to appear in the randomly-generated string(s).")> _
    Public Property AllowNumbers() As Boolean
        Get
            Return _AllowNumbers
        End Get
        Set(ByVal value As Boolean)
            _AllowNumbers = value
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying whether uppercase letters are allowed to appear in the randomly-generated string(s).
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(True), Description("Gets or sets a value specifying whether uppercase letters are allowed to appear in the randomly-generated string(s).")> _
    Public Property AllowUppercaseAlpha() As Boolean
        Get
            Return _AllowUppercaseAlpha
        End Get
        Set(ByVal value As Boolean)
            _AllowUppercaseAlpha = value
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying whether lowercase letters are allowed to appear in the randomly-generated string(s).
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(True), Description("Gets or sets a value specifying whether lowercase letters are allowed to appear in the randomly-generated string(s).")> _
    Public Property AllowLowercaseAlpha() As Boolean
        Get
            Return _AllowLowercaseAlpha
        End Get
        Set(ByVal value As Boolean)
            _AllowLowercaseAlpha = value
        End Set
    End Property


    ''' <summary>
    ''' Gets or sets a value specifying whether the randomly-generated strings should be unique (as a series of raffle tickets drawn from a hat) or not (as a series of die rolls).
    ''' If set to True, the quantity of strings requested must be less than or equal to the number of strings that exist (with the requested length and characters).
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    <Browsable(True), DefaultValue(False), Description("Gets or sets a value specifying whether the randomly-generated strings should be unique (as a series of raffle tickets drawn from a hat) or not (as a series of die rolls).  If set to True, the quantity of strings requested must be less than or equal to the number of strings that exist (with the requested length and characters).")> _
    Public Property EnsureUnique() As Boolean
        Get
            Return _EnsureUnique
        End Get
        Set(ByVal value As Boolean)
            _EnsureUnique = value
        End Set
    End Property


#End Region


    Public Sub New()

    End Sub


    Public Overrides Sub Dispose()
        MyBase.Dispose()


    End Sub


    ''' <summary>
    ''' Connects to the remote service, requests randomly-generated strings that meet the critera specified by this instance's property values, and returns the response as an array of strings.
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function GetStrings() As String()


        Dim NewStrings() As String = Nothing

        Dim FullURL As String = ""
        Dim Response As String = ""
        Dim ResponseElements() As String = Nothing
        Dim TempResponseElement As String


        Try

            'Initialize the return array.
            ReDim NewStrings(-1)

            'Build the full URL path, including all of the variable name/value pairs.
            FullURL = GetFullURL()

            'Request the data from the server.
            Response = RemoteData.GetPageSimple(FullURL, ConnectionTimeout)

            'Does the response contain an error (if so, it will contain a colon, as per the specification at http://random.org/clients/http/ )
            If Response.Contains(":") Then
                Throw New Exception(Response)
            Else

                'Normalize linefeeds/carriage returns.
                Response = Response.Replace(vbCrLf, vbLf).Replace(vbCr, vbLf)

                'Remove trailing linefeeds.
                While Response.EndsWith(vbLf)
                    Response = Response.Substring(0, Response.Length - 1)
                End While

                'Split the response into an array of numeric strings.
                ResponseElements = Split(Response, vbLf)

                'Add each response element to our final array of integers.
                For Each TempResponseElement In ResponseElements
                    If TempResponseElement.Trim() <> "" Then
                        ReDim Preserve NewStrings(NewStrings.Length)
                        NewStrings(NewStrings.Length - 1) = TempResponseElement.Trim()
                    End If
                Next

            End If

        Catch ex As Exception

            Throw

        Finally

            ResponseElements = Nothing

        End Try


        Return NewStrings


    End Function


    Private Function GetFullURL() As String


        Dim sbFullURL As New System.Text.StringBuilder()
        Dim FullURL As String = ""


        Try

            sbFullURL.Append(URL)

            sbFullURL.Append("?num=")
            sbFullURL.Append(Quantity.ToString())

            sbFullURL.Append("&len=")
            sbFullURL.Append(StringLength.ToString())

            sbFullURL.Append("&digits=")
            sbFullURL.Append(IIf(AllowNumbers, "on", "off"))

            sbFullURL.Append("&upperalpha=")
            sbFullURL.Append(IIf(AllowUppercaseAlpha, "on", "off"))

            sbFullURL.Append("&loweralpha=")
            sbFullURL.Append(IIf(AllowLowercaseAlpha, "on", "off"))

            sbFullURL.Append("&unique=")
            sbFullURL.Append(IIf(EnsureUnique, "on", "off"))

            sbFullURL.Append("&format=")
            sbFullURL.Append("plain")

            sbFullURL.Append("&rnd=")
            Select Case RandomizationMethod
                Case RandomizationMethods.IdentifierBased
                    sbFullURL.Append("id.")
                    sbFullURL.Append(RandomizationIdentifier)
                Case RandomizationMethods.DateBased
                    sbFullURL.Append("date.")
                    sbFullURL.Append(RandomizationDate.ToString("yyyy-MM-dd"))
                Case Else
                    sbFullURL.Append("new")
            End Select

            FullURL = sbFullURL.ToString()

        Catch ex As Exception

            Throw

        Finally

            sbFullURL = Nothing

        End Try


        Return FullURL


    End Function


End Class
